<?php
// API endpoint for geoip.coffermedia.com
header('Content-Type: application/json');

require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../lib/GeoIP.php';
require_once __DIR__ . '/../lib/Auth.php';
require_once __DIR__ . '/../lib/RateLimiter.php';

function respond($statusCode, $data) {
    http_response_code($statusCode);
    echo json_encode($data);
    exit;
}

// 1) Get API key
$apiKey = $_GET['key'] ?? '';
if (!$apiKey) {
    respond(401, ['error' => 'API key required']);
}

// 2) Validate key
$auth = new Auth($conn);
$keyInfo = $auth->getKey($apiKey);
if (!$keyInfo) {
    respond(401, ['error' => 'Invalid API key']);
}
if ($keyInfo['status'] !== 'active') {
    respond(403, ['error' => 'API key disabled']);
}
$apiKeyId = (int)$keyInfo['id'];
$dailyLimit = (int)$keyInfo['daily_limit'];

// 3) Determine client IP & query IP
$clientIp = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
$queryIp = $_GET['ip'] ?? $clientIp;

// 4) Rate limiting
$rateLimiter = new RateLimiter($conn, 60); // 60 req/min per key+IP
$rl = $rateLimiter->checkAndIncrement($apiKeyId, $clientIp, $dailyLimit);
if (!$rl['ok']) {
    respond(429, ['error' => 'Rate limit exceeded', 'reason' => $rl['reason']]);
}

// 5) Lookup location
$geo = new GeoIP($conn);
$location = $geo->lookup($queryIp);

// 6) Log usage
$userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
$referer   = $_SERVER['HTTP_REFERER'] ?? '';

$stmt = $conn->prepare("
    INSERT INTO api_logs (api_key_id, client_ip, query_ip, country, region, city, user_agent, referer)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
");
$stmt->bind_param(
    'isssssss',
    $apiKeyId,
    $clientIp,
    $location['ip'],
    $location['country'],
    $location['region'],
    $location['city'],
    $userAgent,
    $referer
);
$stmt->execute();
$stmt->close();

// 7) Success response
respond(200, [
    'success'   => true,
    'ip'        => $location['ip'],
    'country'   => $location['country'],
    'region'    => $location['region'],
    'city'      => $location['city'],
    'latitude'  => $location['latitude'],
    'longitude' => $location['longitude'],
    'timezone'  => $location['timezone'],
    'source'    => $location['source']
]);
?>
